// Shared in-memory storage that persists across all API routes
interface SessionData {
  sessionId: string;
  currentStep: string;
  status: "pending" | "approved" | "rejected";
  data: Record<string, unknown>;
  adminAction?: string;
  rejectionReason?: string;
  redirectTo?: string;
  showEmailOtpModal?: boolean;
  createdAt: Date;
  updatedAt: Date;
}

// Global storage that persists across all API routes
const globalSessions = new Map<string, SessionData>();

export class SessionStorage {
  static set(sessionId: string, session: SessionData) {
    globalSessions.set(sessionId, session);
    console.log(
      `📝 Session stored: ${sessionId} - Step: ${session.currentStep} - Status: ${session.status}`
    );
    console.log(`📊 Total sessions: ${globalSessions.size}`);
  }

  static get(sessionId: string): SessionData | null {
    return globalSessions.get(sessionId) || null;
  }

  static getAll(): SessionData[] {
    return Array.from(globalSessions.values());
  }

  static getAllPending(): SessionData[] {
    return Array.from(globalSessions.values()).filter(
      (session) => session.status === "pending"
    );
  }

  static update(
    sessionId: string,
    updates: Partial<SessionData>
  ): SessionData | null {
    const session = globalSessions.get(sessionId);
    if (!session) return null;

    const updatedSession = {
      ...session,
      ...updates,
      updatedAt: new Date(),
    };

    globalSessions.set(sessionId, updatedSession);
    console.log(
      `🔄 Session updated: ${sessionId} - Status: ${updatedSession.status}`
    );

    if (updates.rejectionReason) {
      console.log(`❌ Rejection reason: ${updates.rejectionReason}`);
    }

    return updatedSession;
  }

  static delete(sessionId: string): boolean {
    return globalSessions.delete(sessionId);
  }

  static clear(): void {
    globalSessions.clear();
  }

  static size(): number {
    return globalSessions.size;
  }
}
