import { NextRequest, NextResponse } from "next/server";
import { ServerAntiBotSystem } from "./lib/server-antibot";

export function middleware(request: NextRequest) {
  const { pathname } = request.nextUrl;

  // Admin route protection (existing logic)
  if (pathname.startsWith("/admin")) {
    const session = request.cookies.get("admin_session")?.value;
    const isLogin = pathname === "/admin/login";

    if (!session && !isLogin) {
      const url = new URL("/login", request.url);
      url.searchParams.set("next", pathname);
      return NextResponse.redirect(url);
    }

    if (session && isLogin) {
      const url = new URL("/dashboard", request.url);
      return NextResponse.redirect(url);
    }
  }

  // Anti-bot protection for API routes
  if (pathname.startsWith("/api/")) {
    // Skip anti-bot for admin routes (they have their own protection)
    if (
      pathname.startsWith("/api/admin/") ||
      pathname.startsWith("/api/auth/")
    ) {
      return NextResponse.next();
    }

    // Apply anti-bot protection to user-facing API routes
    const antiBot = new ServerAntiBotSystem();
    const validation = antiBot.validateRequest(request);

    if (validation.isBot) {
      antiBot.logBlockedRequest(request, validation.reason);
      return antiBot.generateBlockResponse();
    }
  }

  return NextResponse.next();
}

export const config = {
  matcher: ["/admin/:path*", "/api/:path*"],
};


